#  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
#
#  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
#  with the License. A copy of the License is located at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
#  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
#  and limitations under the License.

from ideadatamodel import constants

import ideasdk.app
from ideasdk.auth import TokenService, TokenServiceOptions
from ideasdk.client.evdi_client import EvdiClient
from ideasdk.server import SocaServerOptions
from ideasdk.utils import GroupNameHelper
from ideasdk.client.vdc_client import SocaClientOptions, VirtualDesktopControllerClient

import ideaclustermanager
from ideaclustermanager.app.api.api_invoker import ClusterManagerApiInvoker
from ideaclustermanager.app.projects.projects_service import ProjectsService
from ideaclustermanager.app.accounts.accounts_service import AccountsService
from ideaclustermanager.app.accounts.cognito_user_pool import CognitoUserPool, CognitoUserPoolOptions

from ideaclustermanager.app.auth.api_authorization_service import ClusterManagerApiAuthorizationService
from ideaclustermanager.app.authz.roles_service import RolesService
from ideaclustermanager.app.authz.role_assignments_service import RoleAssignmentsService
from ideaclustermanager.app.accounts.ad_automation_agent import ADAutomationAgent
from ideaclustermanager.app.web_portal import WebPortal
from ideaclustermanager.app.notifications.notifications_service import NotificationsService
from ideaclustermanager.app.snapshots.snapshots_service import SnapshotsService
from ideaclustermanager.app.shared_filesystem.shared_filesystem_service import SharedFilesystemService

from res.clients.ad_sync import ad_sync_client
from res.utils.sssd_utils import SSSDConfigEventSubscriber
import res.exceptions as exceptions

from ideabootstrap.file_system import shared_storage
from ideabootstrap.ssh import restrict_ssh, disable_ssh
from ideabootstrap.sudo import sudoer_secure_path
from ideabootstrap.common import (
    cognito_modules,
    idea_certs,
    amazon_ssm_agent,
    idea_proxy,
    network_interface_tags,
    ebs_volume_tags,
    chronyd,
    idea_service_account,
    cloudwatch_agent,
    motd
)
from ideabootstrap import bootstrap_common
from ideabootstrap.common.constants import (
    INSTANCE_READY_LOCK
)


from typing import Optional
import os
import time


class ClusterManagerApp(ideasdk.app.SocaApp):
    """
    cluster manager app
    """

    def __init__(self, context: ideaclustermanager.AppContext,
                 config_file: str,
                 env_file: str = None,
                 config_overrides_file: str = None,
                 validation_level: int = constants.CONFIG_LEVEL_CRITICAL,
                 **kwargs):

        api_path_prefix = context.config().get_string('cluster-manager.server.api_context_path', f'/{context.module_id()}')
        enable_tls = context.config().get_bool('cluster-manager.server.enable_tls', False)
        cert_key, cert_file = idea_certs.get_cert_key_and_file()

        super().__init__(
            context=context,
            config_file=config_file,
            api_invoker=ClusterManagerApiInvoker(context=context),
            env_file=env_file,
            config_overrides_file=config_overrides_file,
            validation_level=validation_level,
            server_options=SocaServerOptions(
                api_path_prefixes=[
                    api_path_prefix
                ],
                enable_http_file_upload=True,
                enable_tls=enable_tls,
                tls_certificate_file=cert_file,
                tls_key_file=cert_key,
                enable_metrics=True
            ),
            **kwargs
        )
        self.context = context
        self.logger = self.context.logger()
        self.web_portal: Optional[WebPortal] = None

    def run_bootstrap(self):
        if os.path.isfile(INSTANCE_READY_LOCK):
            self.logger.info(f"Configuration was already completed, skipping. INSTANCE_READY_LOCK file: {INSTANCE_READY_LOCK}")
            return

        bootstrap_common.set_reboot_required("no")

        # Configure SSH and Sudoer
        restrict_ssh.configure()
        disable_ssh.configure()
        sudoer_secure_path.configure()

        if os.environ.get('IDEA_HTTPS_PROXY'):
            idea_proxy.set_proxy()

        idea_service_account.setup_account()
        amazon_ssm_agent.configure()
        cloudwatch_agent.setup()

        network_interface_tags.setup()
        ebs_volume_tags.setup()

        bootstrap_common.disable_se_linux()
        chronyd.configure()
        bootstrap_common.disable_ulimit()
        bootstrap_common.disable_strict_host_check()
        motd.disable_update()

        messages = [f'{os.environ.get("IDEA_MODULE_NAME", "")} (v{os.environ.get("IDEA_MODULE_VERSION", "")}), Cluster: {os.environ.get("IDEA_CLUSTER_NAME", "")}']
        motd.update(messages)

        # Mount shared storage
        shared_storage.configure()

        idea_certs.setup()

        # Cognito Setup
        cognito_modules.configure()

        timestamp = str(int(time.time()))
        with open(INSTANCE_READY_LOCK, 'w') as f:
            f.write(timestamp)

        bootstrap_common.check_reboot_required()

    def soca_app_warmup(self):
        self.run_bootstrap()

    def soca_app_initialize(self):
        # Start SSSD service
        sssd_config_event_subscriber = SSSDConfigEventSubscriber(
            self.logger,
            self.context.module_id(),
        )
        self.context.config().subscribe(
            sssd_config_event_subscriber
        )
        sssd_config_event_subscriber.restart_sssd_service()

        # group name helper
        self.context.group_name_helper = GroupNameHelper(self.context)
        administrators_group_name = self.context.group_name_helper.get_cluster_administrators_group()
        managers_group_name = self.context.group_name_helper.get_cluster_managers_group()

        # token service
        domain_url = self.context.config().get_string('identity-provider.cognito.domain_url', required=True)
        provider_url = self.context.config().get_string('identity-provider.cognito.provider_url', required=True)
        client_id = self.context.config().get_secret('cluster-manager.client_id', required=True)
        client_secret = self.context.config().get_secret('cluster-manager.client_secret', required=True)
        vdc_module_id = self.context.config().get_module_id(constants.MODULE_VIRTUAL_DESKTOP_CONTROLLER)
        self.context.token_service = TokenService(
            context=self.context,
            options=TokenServiceOptions(
                cognito_user_pool_provider_url=provider_url,
                cognito_user_pool_domain_url=domain_url,
                client_id=client_id,
                client_secret=client_secret,
                client_credentials_scope=[
                    f'{os.environ.get("IDEA_CLUSTER_NAME", "")}-{vdc_module_id}/read',
                    f'{os.environ.get("IDEA_CLUSTER_NAME", "")}-{vdc_module_id}/write',
                ],
                administrators_group_name=administrators_group_name,
                managers_group_name=managers_group_name
            )
        )

        # cognito user pool
        user_pool_id = self.context.config().get_string('identity-provider.cognito.user_pool_id', required=True)
        self.context.user_pool = CognitoUserPool(
            context=self.context,
            options=CognitoUserPoolOptions(
                user_pool_id=user_pool_id,
                admin_group_name=administrators_group_name,
                client_id=client_id,
                client_secret=client_secret
            )
        )

        # accounts service
        ds_provider = self.context.config().get_string('directoryservice.provider', required=True)

        if ds_provider in {constants.DIRECTORYSERVICE_AWS_MANAGED_ACTIVE_DIRECTORY, constants.DIRECTORYSERVICE_ACTIVE_DIRECTORY}:
            self.context.ad_automation_agent = ADAutomationAgent(
                context=self.context,
            )

        # account service
        evdi_client = EvdiClient(self.context)
        self.context.accounts = AccountsService(
            context=self.context,
            user_pool=self.context.user_pool,
            evdi_client=evdi_client,
            token_service=self.context.token_service
        )

        # roles service
        self.context.roles = RolesService(
            context=self.context
        )

        # role assignments service
        self.context.role_assignments = RoleAssignmentsService(
            context=self.context
        )

        #api authorization service
        self.context.api_authorization_service = ClusterManagerApiAuthorizationService(
            accounts=self.context.accounts,
            config=self.context.config(),
            roles=self.context.roles,
            role_assignments=self.context.role_assignments
        )

        internal_endpoint = self.context.config().get_cluster_internal_endpoint()
        self.context.vdc_client = VirtualDesktopControllerClient(
                context=self.context,
                options=SocaClientOptions(
                    endpoint=f'{internal_endpoint}/{vdc_module_id}/api/v1',
                    enable_logging=False,
                    verify_ssl=False),
                token_service=self.context.token_service
            )

        self.context.snapshots = SnapshotsService(
            context=self.context
        )

        # projects service
        self.context.projects = ProjectsService(
            context=self.context,
            accounts_service=self.context.accounts,
            vdc_client=self.context.vdc_client
        )

        # notifications
        self.context.notifications = NotificationsService(
            context=self.context,
            accounts=self.context.accounts
        )

        self.context.shared_filesystem = SharedFilesystemService(
            context=self.context
        )

        # web portal
        self.web_portal = WebPortal(
            context=self.context,
            server=self.server
        )
        self.web_portal.initialize()

    def soca_app_start(self):
        if self.context.ad_automation_agent is not None:
            self.context.ad_automation_agent.start()

        self.context.notifications.start()

        try:
            self.context.distributed_lock().acquire(key='initialize-defaults')

            ad_sync_client.start_ad_sync()
        except (exceptions.ADSyncConfigurationNotFound, exceptions.ADSyncInProcess):
            # AD configuration may not be provided yet when the Cluster Manager starts
            # AD Sync may have been triggered by the scheduler Lambda and is still in progress
            pass
        finally:
            self.context.distributed_lock().release(key='initialize-defaults')

    def soca_app_stop(self):

        if self.context.ad_automation_agent is not None:
            self.context.ad_automation_agent.stop()

        if self.context.notifications is not None:
            self.context.notifications.stop()
